/* ***************************************************************************+
 * ITX package (cnrg.itx) for telephony application programming.              *
 * Copyright (c) 1999  Cornell University, Ithaca NY.                         *
 * A copy of the license is distributed with this package.  Look in the docs  *
 * directory, filename GPL.  Contact information: bergmark@cs.cornell.edu     *
 ******************************************************************************/

package cnrg.itx.gtwy;
import java.util.Vector;
import cnrg.itx.datax.*;

/**
 * This is abstracted as the source for the voice data packets sent from the Gateway to the computer
 * network.
 * 
 * @author James Wann
 * @version 1.0a
 * @see RecordChannel
 * @see cnrg.itx.datax.Source
 */
public class RecordSource implements Source
{
	/**
	 * The Channel instance where RecordSource sends the voice data to.
	 * NOTE: This is not to be confused with the Gateway channels!
	 */
	private Channel myChannel;
	
	/**
	 * The Line instance related to RecordSource.
	 */
	private Line myLine;
	
	/**
	 * true if source is able to send voice data; false otherwise.
	 */
	private boolean isOpen;
	
	/**
	 * true if source is muted; false otherwise.
	 */
	private boolean isMute;
	
	/**
	 * Number of bytes written to the computer network.
	 */
	private int bytesWritten = 0;
	
	/**
	 * Number of packets sent to the computer network.  Size equal to transfer size from phone to
	 * Gateway.
	 */
	private int iNumPackets = 0;
	
	/**
	 * @param c the <code>Channel</code> instance used for sending packets from a phone to the computer
	 * network.
	 * @param l the <code>Line</code> instance associated with <code>RecordSource</code>.
	 * @see Line
	 * @see cnrg.itx.datax.Channel
	 */
	public RecordSource(Channel c, Line l)
	{
		myChannel = c;
		myLine = l;
		isOpen = false;
		isMute = false;
	}
	
	/**
	 * Method to start the source.
	 */
	public void start() 
	{
		isOpen = true;
	}
	
	/**
	 * Method to stop the source.
	 */
	public void stop() 
	{
		isOpen = false;
	}
	
	/**
	 * Method to mute and unmute the source.
	 * 
	 * @param mode true if muted; false otherwise.
	 * @return true if muted; false otherwise.
	 */
	public boolean mute(boolean mode)
	{
		isMute = mode;
		return isMute;
	}
	
	/**
	 * Method for closing the source.
	 */
	public void close()
	{
		stop();
	}
    
	/**
	 * Called whenever data is received from a phone.  Sends data to the computer network.
	 * 
	 * @param data the voice data in bytes
	 */
	public void sendToNetwork(byte[] data)
	{	
		if (isOpen && !isMute)
		{
			myChannel.push(data);
			bytesWritten += data.length;
			iNumPackets++;
		}
	}
	
	/**
	 * Called to get the <code>Line</code> instance associated with <code>RecordSource</code>.
	 * 
	 * @return the <code>Line</code> instance.
	 * @see Line
	 */
	public Line getLine()
	{
		return myLine;
	}
	
	/**
	 * The Gateway is not concerned about properties.  Returns null.
     */ 
	public PropertiesCollection getProperties() throws DataException
    {
		return null;
    }

    /**
     * No properties are set.
     */
    public void setProperties(PropertiesCollection pc) throws DataException
    {
    }

    /**
     * Peer properties are never set.
     */
    public void setPeerProperties(PropertiesCollection pc) throws DataException
    { 
    }

    /**
     * @return the Stats object which describes the current state of the source.  
	 */
	public Stats getStatistics()
	{
		Stats networkStats = new Stats();
		int lineNumber = myLine.getLineNumber();
		
		networkStats.addStat("Device", "Gateway Line Source");
		networkStats.addStat("<Gateway Line Source " + lineNumber + "> Bytes written to channel", new Integer(bytesWritten));
		networkStats.addStat("<Gateway Line Source " + lineNumber + "> Number Packets Received from wire", new Integer(iNumPackets)); 
	   
		return networkStats;
	}
}